using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.UI;


#if UNITY_EDITOR
using UnityEditor;
#endif

namespace CHTLand.SDK.Module.Component
{
    public class ChtBaseComponent : MonoBehaviour
    {
            protected enum ComponentType
            {
                ChtModelComponent,
                ChtDialogComponentWithAvatarNPC,
                ChtChestComponent,
                ChtDialogComponentWithCustomNPC,
                ChtDialogComponentOnlyConversation,
                ChtFrameComponent,
                ChtTeleportComponent,
                ChtLotteryComponent,
                ChtSeatComponent,
                ChtEntryComponent,
                ChtPortalComponent,
                ChtClimbableComponent
            }
            
            [SerializeField, HideInInspector]
            protected string objectID;
            public string ObjectID => objectID;
            
            public void GenerateID()
            {
            #if UNITY_EDITOR
                if(this == null || gameObject == null)
                    return;
                objectID = System.Guid.NewGuid().ToString("N").Substring(0,6);
                Debug.LogWarning($"[ID Created]{gameObject.name} => ID: {objectID}",this);
            #endif
            }
            protected virtual void Awake()
            {
                if (string.IsNullOrEmpty(objectID))
                {
                    GenerateID();
                }
            }
            #if UNITY_EDITOR
            protected virtual void OnValidate()
            {
                if (!Application.isPlaying)
                {
                    string currentObjectID = objectID;

                    if (!string.IsNullOrEmpty(currentObjectID))
                    {
                        // 檢查場景中是否有其他物件已經使用這個 ID
                        var allComponents = GameObject.FindObjectsOfType<ChtBaseComponent>(true);
                        
                        foreach (var comp in allComponents)
                        {
                            if (comp != this && comp.ObjectID == this.objectID)
                            {
                                Debug.LogWarning($"[Detect Duplicate]:{gameObject.name}");
                                GenerateID();
                                EditorUtility.SetDirty(this);
                                break;
                            }
                        }
                    }

                }
            }
            #endif
            
            #if UNITY_EDITOR

            protected MeshFilter meshFilter; 
            protected MeshRenderer meshRenderer;
            protected Material material;
            protected BoxCollider boxCollider;
            protected bool isDrawICon = true;
            protected Vector3 size;
            protected Vector3 center;
            private Collider myCollider;

            //Color Palette
            protected Color blue = new UnityEngine.Color(88.0f / 255.0f, 178.0f / 255.0f, 220.0f / 255.0f, 150.0f / 255.0f);
            protected Color purple = new UnityEngine.Color(112.0f/255.0f,100.0f/255f,154.0f/255.0f,210.0f/255.0f);
            protected Color red = new UnityEngine.Color(235.0f / 255.0f, 122.0f / 255.0f, 119.0f / 255.0f, 150.0f / 255.0f);
            protected Color yellow = new UnityEngine.Color(246.0f / 255.0f, 197.0f / 255.0f, 85.0f / 255.0f, 150.0f / 255.0f);
            protected Color green = new UnityEngine.Color(134.0f / 255.0f, 193.0f / 255.0f, 102.0f / 255.0f, 200f / 255.0f);
            
            //Path 
            protected string BackPrefab = "Editor/Design/Prefab/Back.prefab";
            protected string TargetStandOn = "Editor/Design/Prefab/TargetStandOn.prefab";
            
            protected virtual Mesh GetBuiltinQuadMesh()
            {
                GameObject tempCube = GameObject.CreatePrimitive(PrimitiveType.Quad);
                Mesh mesh = tempCube.GetComponent<MeshFilter>().sharedMesh;
                DestroyImmediate(tempCube);
                return mesh;  
            }
            protected virtual Mesh GetBuiltinCylinderMesh()
            {
                GameObject tempCube = GameObject.CreatePrimitive(PrimitiveType.Cylinder);
                Mesh mesh = tempCube.GetComponent<MeshFilter>().sharedMesh;
                DestroyImmediate(tempCube);
                return mesh;  
            }
            //取得unity內建cube的mesh，不需改變cube的tramsfor 
            protected virtual Mesh GetBuiltinCubeMesh()
            {
                GameObject tempCube = GameObject.CreatePrimitive(PrimitiveType.Cube);
                Mesh mesh = tempCube.GetComponent<MeshFilter>().sharedMesh;
                DestroyImmediate(tempCube);
                return mesh;  
            }

            //取得unity內建cube的mesh，需要cube的tramsform
            protected virtual void ApplyCubeMeshAndTransform(Vector3? scale = null, Vector3? position = null, Quaternion? rotation = null)
            {
                // 創一個臨時的立方體
                GameObject tempCube = GameObject.CreatePrimitive(PrimitiveType.Cube);

                // 設置臨時立方體的 Transform 參數
                tempCube.transform.position = position ?? transform.position;
                tempCube.transform.rotation = rotation ?? transform.rotation;
                tempCube.transform.localScale = scale ?? transform.localScale;

                // 獲取立方體的 Mesh 並應用到掛載腳本的物體上
                Mesh mesh = tempCube.GetComponent<MeshFilter>().sharedMesh;
                MeshFilter targetMeshFilter = gameObject.GetComponent<MeshFilter>();
                if (targetMeshFilter == null)
                {
                    targetMeshFilter = gameObject.AddComponent<MeshFilter>();
                }
                targetMeshFilter.sharedMesh = mesh;

                // 將臨時立方體的 Transform 應用到當前物體上
                transform.position = tempCube.transform.position;
                transform.rotation = tempCube.transform.rotation;
                transform.localScale = tempCube.transform.localScale;

                DestroyImmediate(tempCube);
                
            }

            //創建MeshFilter
            protected virtual void CreateMeshFilter(string meshType)
            {
                if (meshFilter == null)
                {
                    meshFilter = gameObject.GetComponent<MeshFilter>();
                    if (meshFilter == null)
                    {
                        meshFilter = gameObject.AddComponent<MeshFilter>();
                        
                        if (meshType == "Cube")
                        {
                            meshFilter.sharedMesh = GetBuiltinCubeMesh();
                        }
                        else if (meshType == "Cylinder")
                        {
                            meshFilter.sharedMesh = GetBuiltinCylinderMesh();
                        }
                        else
                        {
                            meshFilter.sharedMesh = GetBuiltinQuadMesh();
                        }
                         
                    }
                    
                }
            }
            
            //創建Renderer
            protected virtual void CreateRenderer()
            {
                //default 版位藍色
                UnityEngine.Color defaultColor = new UnityEngine.Color(88.0f / 255.0f, 178.0f / 255.0f, 220.0f / 255.0f, 150.0f / 255.0f);
                meshRenderer = gameObject.GetComponent<MeshRenderer>();
                
                if (meshRenderer == null)
                {
                    meshRenderer = gameObject.AddComponent<MeshRenderer>();

                    string compMaterialPath = "Editor/Design/TransparentComp.mat";
                    string addAssetsPath = "Assets/" + compMaterialPath;

                    if (System.IO.File.Exists(addAssetsPath))
                    {
                        material = AssetDatabase.LoadAssetAtPath<Material>(addAssetsPath);
                        
                        if (material != null)
                        {
                            meshRenderer.sharedMaterial = material;
                            material.color = defaultColor;
                        }
                        else
                        {
                            meshRenderer.sharedMaterial = new Material(Shader.Find("Standard"));
                            meshRenderer.sharedMaterial.SetFloat("_Mode", 3);
                        }
                    } 
                    else
                    {
                        var packagePath = "Packages/com.cht.cht-land/" + compMaterialPath;
                        material = AssetDatabase.LoadAssetAtPath<Material>(packagePath);
                        
                        if (material != null)
                        {
                            material.color = defaultColor;
                            meshRenderer.sharedMaterial = material;
                        }
                        else
                        {
                            meshRenderer.sharedMaterial = new Material(Shader.Find("Standard"));
                            meshRenderer.sharedMaterial.SetFloat("_Mode", 3);
                            
                        }
                    }
                    
                    
                }
            }

            //Material套用至個別子物件
            protected virtual void ApplyChildMaterial(string materialPath, string childPath)
            {
                var material = AssetDatabase.LoadAssetAtPath<Material>(materialPath);
                if (material == null)
                {
                    Debug.Log("No Material Path!");
                }
                else
                {
                    //var parentModel = transform.Find(parentModelName);
                    var childModel = transform.Find(childPath);
                    if (childModel == null)
                    {
                        Debug.Log("Can't find parent or child Model!");
                        return;
                    }
                    else
                    {
                        var meshRenderer = childModel.GetComponent<MeshRenderer>();
                        meshRenderer.sharedMaterial = material;
                    }
                }                                        
            }

            //創建Collider
            protected virtual void CreateCollider()
            {
                boxCollider  = this.gameObject.GetComponent<BoxCollider>();
                if(boxCollider == null)
                {
                    boxCollider = this.gameObject.AddComponent<BoxCollider>();
                }
            }

            protected virtual void InheritCreateCamera()
            {
                Camera camera = transform.GetComponentInChildren<Camera>();
                if(camera == null)
                {
                    //創建Camera
                    GameObject childCamera = new GameObject("NpcCamera");
                    childCamera.transform.parent = this.transform;
                    childCamera.transform.localPosition  = Vector3.zero;
                    childCamera.AddComponent<Camera>();

                    //設定Camera Default參數
                    var cameraComponent = childCamera.GetComponent<Camera>();
                    cameraComponent.clearFlags = CameraClearFlags.SolidColor;
                    cameraComponent.cullingMask = 1 << LayerMask.NameToLayer("NPC");
                    cameraComponent.fieldOfView = 35.0f;
                    cameraComponent.targetDisplay = 1;

                    //設定Camera Default位置
                    Vector3 defaultPosition = new Vector3(0.0f,1.8f,2.0f);
                    Quaternion defaultQuaternion = Quaternion.Euler(15.0f, 180.0f, 0.0f);
                    childCamera.transform.localPosition = defaultPosition;
                    childCamera.transform.rotation = defaultQuaternion; 

                }
            }
            
             //設定Camera的TargetTexture
            protected virtual void InheritSetCameraTargetTexture(RenderTexture cameraTargetTexture)
            {
                var cameraTransform = transform.Find("NpcCamera");
                if(cameraTransform != null)
                {
                    var cameraComponent = cameraTransform.GetComponent<Camera>();
                    if(cameraComponent != null)
                    {
                        cameraComponent.targetTexture = cameraTargetTexture;
                    }
                    else
                    {
                        Debug.Log("Not Find Camera Component.");
                    }
                } 
                else
                {
                    Debug.Log("Not Find NPC Camera");
                }
                Debug.Log("Create Render Texture!");
            }
            
            //設定NpcCameraView的RawImage的texture
            protected void MySetRawImageTexture(RenderTexture rawImageTexture)
            {
                var rawImage = transform.Find("NpcCameraView/RawImage");
                var rawImageComponent = rawImage.GetComponent<RawImage>();
                if(rawImageComponent != null)
                {
                    rawImageComponent.texture = rawImageTexture;
                }
                else
                {
                    Debug.Log("Not Find RawImage");
                } 
            }

                
            //鎖定scale固定比例
            protected Vector3 previousScale;
            protected ComponentType componentType;

            protected virtual void LockScale()
            {
                //Debug.Log($"Current component type: {componentType}");
                switch (componentType)
                {
                    case ComponentType.ChtModelComponent:
                        LockScaleCube();
                        break;
                    case ComponentType.ChtFrameComponent:
                        LockScaleFrame();
                        break;
                    case ComponentType.ChtSeatComponent:
                    case ComponentType.ChtEntryComponent:
                    case ComponentType.ChtPortalComponent:
                        LockScaleSame();
                        break;
                    default:
                        LockScaleCube();
                        break;
                }
            }
            
            //鎖定Scale的3軸為相同
            protected virtual void LockScaleCube()
            {
                float scaleFactor = transform.localScale.x;
                transform.localScale = new Vector3(scaleFactor, scaleFactor, scaleFactor);
            }
            //鎖定Scale的 x 與 y 為16:9
            protected virtual void LockScaleFrame()
            {
                float scaleFactor_x = transform.localScale.x;
                float scaleFactor_y = transform.localScale.y;

                if (Mathf.Abs(scaleFactor_x - previousScale.x) > Mathf.Abs(scaleFactor_y - previousScale.y))
                {
                    // X 軸變化比較大时，調 Y 軸
                    transform.localScale = new Vector3(scaleFactor_x, scaleFactor_x * 9f / 16f, transform.localScale.z);
                }
                else
                {
                    // Y 軸變化比較大时，調 X 軸
                    transform.localScale = new Vector3(scaleFactor_y * 16f / 9f, scaleFactor_y, transform.localScale.z);
                }
            }
            //鎖定Scale為固定值
            protected virtual void LockScaleSame()
            {
                if (transform.localScale != Vector3.one)
                {
                    transform.localScale = Vector3.one;
                }   
            }

            private MaterialPropertyBlock materialPropertyBlock;
            /// <summary>
            /// 檢查是否碰到其他物件
            /// </summary>
            /// <param name="defaultColor"></param>
            /// <param name="changeColor"></param>
            /// <param name="Collidersize"></param>
            /// <param name="MyColliderCenter"></param>
            /// <param name="ChildObjectName"></param> <summary>
            protected virtual void DetectCollisions(UnityEngine.Color? defaultColor = default, UnityEngine.Color? changeColor = default, Vector3?Collidersize = default, Vector3?MyColliderCenter = default, string ChildObjectName = null)
            {
                defaultColor = defaultColor ?? new UnityEngine.Color(88.0f / 255.0f, 178.0f / 255.0f, 220.0f / 255.0f, 150.0f / 255.0f);
                changeColor = changeColor ?? new UnityEngine.Color(235.0f / 255.0f, 122.0f / 255.0f, 119.0f / 255.0f, 150.0f / 255.0f);
                Vector3 boxSize = Collidersize ?? transform.localScale;
                Vector3 boxCenter = MyColliderCenter ?? transform.position;
                
                materialPropertyBlock = new MaterialPropertyBlock();
                materialPropertyBlock.SetColor("_Color", defaultColor.Value);
                
                //設為defaultColor藍色
                if (ChildObjectName == null)
                {
                    meshRenderer = gameObject.GetComponent<MeshRenderer>();
                    meshRenderer.SetPropertyBlock(materialPropertyBlock);
                }
                else if (ChildObjectName != null)
                {
                    transform.Find(ChildObjectName).gameObject.GetComponent<MeshRenderer>().SetPropertyBlock(materialPropertyBlock);
                }
                else
                {
                    Debug.Log("ChildObjectName is null!");
                }
                
                myCollider = gameObject.GetComponent<BoxCollider>();
                Collider[] hitColliders = Physics.OverlapBox(boxCenter, boxSize / 2, transform.rotation, ~0);

                // 根据碰撞结果切换颜色
                if (hitColliders.Length > 0)
                {
                    foreach (Collider collider in hitColliders)
                    {
                        if (collider != myCollider) //避免偵測自己的Collider
                        {
                            if (ChildObjectName == null)
                            {
                                Debug.Log("Detected collision with " + hitColliders[0].name);
                                materialPropertyBlock.SetColor("_Color", changeColor.Value);
                                meshRenderer.SetPropertyBlock(materialPropertyBlock);
                            }
                            else
                            {
                                if (collider != transform.Find(ChildObjectName).gameObject.GetComponent<BoxCollider>())
                                {
                                    Debug.Log("Detected collision with " + hitColliders[0].name);
                                    materialPropertyBlock.SetColor("_Color", changeColor.Value);
                                    transform.Find(ChildObjectName).gameObject.GetComponent<MeshRenderer>().SetPropertyBlock(materialPropertyBlock);
                                }
                            }

                        }
                    }
                }
            }
            
            //繪製Wireframe外框線
            protected virtual void DrawCubeIcon(Vector3? size = null, Vector3? center = null, Quaternion? rotations = null)
            {
                Vector3 boxSize = size ?? transform.localScale;
                Vector3 position = center ?? transform.position;
                Quaternion rotation = rotations ?? transform.rotation;

                //繪製cube的外框線
                Gizmos.color = new UnityEngine.Color(112.0f/255.0f,100.0f/255f,154.0f/255.0f,210.0f/255.0f);
                
                //使用cube的大小設定gizmos的矩陣
                Gizmos.matrix = Matrix4x4.TRS(position, rotation, boxSize);
                Gizmos.DrawWireCube(Vector3.zero, Vector3.one);
            }
            //在畫gizmos的時候，將Local的路徑轉成package路徑
            protected void IconPathToPackage(string IconPathWithoutAssets, Vector3 IconPosition)
            {
                var iconPath = "Assets/" + IconPathWithoutAssets;
                if(System.IO.File.Exists(iconPath))
                {
                    Gizmos.DrawIcon(IconPosition, iconPath, true);
                }
                else
                {
                    var packageIconPath = "Packages/com.cht.cht-land/" + IconPathWithoutAssets;
                    Gizmos.DrawIcon(IconPosition, packageIconPath, true);
                } 
            }

            protected float toDistance;
            protected float closeDistance = 10f;
    
            //視角的攝像機離多遠
            protected virtual void CameraToSceneView()
            {
                if (SceneView.lastActiveSceneView != null)
                {
                    Camera sceneCamera = SceneView.lastActiveSceneView.camera;
                    float distance = Vector3.Distance(sceneCamera.transform.position, transform.position);
                    toDistance = distance;
                }
            }         
   
            /// <summary>
            /// 生成一個Prefab
            /// </summary>
            /// <param name="prefabPath"></param>
            protected virtual void FindPrefab(string prefabPath)
            {
                string addAssets = "Assets/" + prefabPath;
                var prefab = AssetDatabase.LoadAssetAtPath<GameObject>(addAssets);

            	if (prefab != null)
            	{
                	PrefabUtility.InstantiatePrefab(prefab, transform);
                	Debug.Log("Prefab loaded!");
					return;
                }

				//如果是Package路徑，先找com.cht.cht-land
				string packagePrefabPath = "Packages/com.cht.cht-land/" + prefabPath;
				var packagePrefab = AssetDatabase.LoadAssetAtPath<GameObject>(packagePrefabPath);
				if (packagePrefab != null)
				{
					PrefabUtility.InstantiatePrefab(packagePrefab, transform);
					Debug.Log("Prefab loaded from com.cht.cht-land!");
					return;
				}
				
				//com.cht.cht-land找不到，再找com.cht.cht-land.prod
				string prodPackagePrefabPath = "Packages/com.cht.cht-land.prod/" + prefabPath;
				var prodPackagePrefab = AssetDatabase.LoadAssetAtPath<GameObject>(prodPackagePrefabPath);
				if (prodPackagePrefab != null)
				{
					PrefabUtility.InstantiatePrefab(prodPackagePrefab, transform);
					Debug.Log("Prefab loaded from com.cht.cht-land.prod!");
					return;
				}
			
			}
            /// <summary>
            /// 當實例被複製時，刪除創建的Prefab避免HideFlag等參數因複製失效
            /// </summary>
            /// <param name="PrefabName"></param>
            protected virtual void InitCopyPrefab(string prefabName, object option, string prefabParentName = null)
            {   
                var prefab = string.IsNullOrEmpty(prefabParentName)
                    ? transform.Find(prefabName)
                    : transform.Find(prefabParentName + "/" + prefabName);               

                if (prefab != null)
                {
                    switch (option)
                    {
                        case HideFlags hideFlags:
                            prefab.hideFlags = hideFlags;
                            break;
                        default:
                            prefab.gameObject.hideFlags = HideFlags.NotEditable;
                            break;
                    }
                    Debug.Log($"[Copy] Detect Prefab Copied, reset {option}");
                }
            }
            /// <summary>
            /// 改變所有子物件的Layer
            /// </summary>
            /// <param name="obj"></param>
            /// <param name="newLayer"></param>
            protected virtual void InheritSetLayerRecursively(GameObject obj, int newLayer)
        {
            if (obj == null)
            {
                return;
            }
            else
            {
                obj.layer = newLayer;
            }
            // 設定子物件的Layer
            foreach (Transform child in obj.transform)
            {
                if (child == null)
                    continue;
                InheritSetLayerRecursively(child.gameObject, newLayer);
            }
        }
            /// <summary>
            /// 設定GameObject為不可編輯
            /// </summary>
            /// <param name="obj"></param>
            protected virtual void InheritLockAllComponents(GameObject obj)
            {
                // 設定 GameObject 本身不可編輯
                obj.hideFlags = HideFlags.HideInHierarchy;

                // 設定子組件不可編輯
                foreach (Transform child in obj.transform)
                {
                    if (child == null)
                        continue;
                    InheritLockAllComponents(child.gameObject);
                }
            }
            
            /// <summary>
            /// 重命名名為 "TargetStandOn" 的子物件，並避免重複命名
            /// </summary>
            protected void RenameChildObjects()
            {
                Transform[] targetChildren = transform.GetComponentsInChildren<Transform>();
                int counter = 0;

                foreach (Transform child in targetChildren)
                {
                    if (child.name == "TargetStandOn")
                    {
                        if (counter == 0)
                        {
                            child.name = "TargetStandOn";
                            counter++;
                            var meshFilter = child.gameObject.GetComponent<MeshFilter>();
                            var renderer = child.gameObject.GetComponent<MeshRenderer>();
                            meshFilter.hideFlags = HideFlags.NotEditable;
                            renderer.hideFlags = HideFlags.NotEditable;
                        
                        }
                        else
                        {
                            child.name = $"TargetStandOn ({counter})";
                            counter++;
                            var meshFilter = child.gameObject.GetComponent<MeshFilter>();
                            var renderer = child.gameObject.GetComponent<MeshRenderer>();
                            meshFilter.hideFlags = HideFlags.NotEditable;
                            renderer.hideFlags = HideFlags.NotEditable;
                            Debug.Log("Renamed child object: " + child.name);
                        }

                    }
                }
            }

            #endif
    }
}