using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using System;

#if UNITY_EDITOR
using UnityEditor;
#endif

namespace CHTLand.SDK.Module.Component
{
    /// <summary>
    /// 爬牆元件
    /// </summary>
    [ExecuteInEditMode]
    public class ChtClimbableComponent : ChtBaseComponent
    {
        [SerializeField] 
        private string climbableComponentID;
        public string ClimbableComponentID => climbableComponentID;

        //Type of component
        [HideInInspector]
        ComponentType type = ComponentType.ChtClimbableComponent;
        
        //Default vertices
        [SerializeField, HideInInspector]
        private Vector3[] vertices;
        [HideInInspector]
        public Vector3[] worldVertices = new Vector3[8]
        {
            Vector3.zero, Vector3.zero, Vector3.zero, Vector3.zero,
            Vector3.zero, Vector3.zero, Vector3.zero, Vector3.zero
        };
        [Range(0, 40)] public float angle; //using this
        public float offsetOnPlaneY; //using this
        [SerializeField] public float standOnPlaneY; //using this 
        public GameObject plane;
        protected override void Awake()
        {
            base.Awake();
            //Debug.LogWarning($"Chest Component ID : {ObjectID}");
        }
        void Start()
        {
            if (Application.isPlaying && CHTLandProxy.componentManager != null)
            {
                CHTLandProxy.componentManager.InitClimbableComponent(this);
            }
            Debug.Log(type.ToString());
        }

#if UNITY_EDITOR

        //Prefab Path
        private string myClimbQuad = "Editor/Design/Prefab/ClimbQuad.prefab";
        private string myTargetStandOn = "Editor/Design/Prefab/TargetStandOn.prefab";
        private MeshFilter myMeshFilter;
        
        //Face and Vertices
        private Vector3[] bottomFace;
        private Vector3[] leftFace;
        private Vector3[] rightFace;
        private Vector3[] frontFace;
        private Vector3[] backFace;
        private Vector3[] topFace;
        private Vector3[][] faces;
        
        //Previous data
        private Vector3[] privousVertices;
        private Quaternion previousRotation;
        private Vector3 previousScales;
        private Vector3 previousPosition;
        private float previousAngle;
        private float lastPlaneY = float.NaN;
        
        //Prefab Transform init
        private Vector3 targetStandOnScale = new Vector3(0.25f, 0.01f, 0.25f);
        private Vector3 myBackPrefabBackPosition = new Vector3(0.51f, 0, 0);
        private Quaternion myBackPrefabBackRotation = Quaternion.Euler(0, 270, 0);
        private Vector3 myBackPrefabScale = new Vector3(0.5f, 0.5f ,0.5f);
        
        //bool and condition
        private bool Initialize = false;
        private bool AngleChanged = false;
        private bool isRotate = false;
        private bool isScale = false;
        private bool isMove = false;
        private bool isInitPrefabPos = false;
        //private bool isChangeColor = false;
        private int updateMesh = 0;

        protected override void OnValidate()
        {
            base.OnValidate();
            if (!Application.isPlaying)
            {
                // 同步 base 的 objectID 給 seatComponentID
                climbableComponentID = objectID;
                UnityEditor.EditorUtility.SetDirty(this); // 確保 Inspector 顯示更新
            }
            //在inspector掛載平面時，取得平面y值。如果掛載plane後移動plane，在Update更新y值
            if (plane != null)
            {
                standOnPlaneY = plane.transform.position.y;
            }
        }

        void OnEnable()
        {
            EditorApplication.update += OnEditorUpdate;
        }
        void OnDisable()
        {
            EditorApplication.update -= OnEditorUpdate;
        }
        void OnEditorUpdate()
        {
            // 如果正在Play或準備Play，就不要做任何事
            if (EditorApplication.isPlayingOrWillChangePlaymode)
            {
                return;
            }
            // 限制Y軸的最小縮放值為4，牆體最低高度應該高於4
            if (transform.localScale.y < 4f)
            {
                Vector3 currentScale = transform.localScale;
                currentScale.y = 4f;
                transform.localScale = currentScale;
            }

            //找到ClimbQuad Prefab
            Transform myClimbQuadTransform = transform.Find("ClimbQuad");
            if (myClimbQuadTransform == null)
            {
                FindPrefab(myClimbQuad);
                myClimbQuadTransform = transform.Find("ClimbQuad");
                myMeshFilter = myClimbQuadTransform.GetComponent<MeshFilter>();
                if (myMeshFilter == null)
                {
                    Debug.LogError("MeshFilter component not found.");
                    return;
                }
            }
            //找到TargetStandOn Prefab
            Transform myTargetStandOnTransform = transform.Find("TargetStandOn");
            if (myTargetStandOnTransform == null)
            {
                FindPrefab(myTargetStandOn);
                myTargetStandOnTransform = transform.Find("TargetStandOn");
                myTargetStandOnTransform.localScale = targetStandOnScale;
            }
            //找到Back Prefab
            Transform myBackPrefabTransform = transform.Find("Back");
            if (myBackPrefabTransform == null)
            {
                FindPrefab(BackPrefab);
                myBackPrefabTransform = transform.Find("Back");
                myBackPrefabTransform.localPosition = myBackPrefabBackPosition;
                myBackPrefabTransform.rotation = myBackPrefabBackRotation;
            }
            //關閉物件編輯
            if (myClimbQuadTransform != null && myTargetStandOnTransform != null && myBackPrefabTransform != null)
            {
                InheritLockAllComponents(myClimbQuadTransform.gameObject);
                InheritLockAllComponents(myTargetStandOnTransform.gameObject);
                InheritLockAllComponents(myBackPrefabTransform.gameObject);
            }

            //初始化頂點後，調整Prefab位置到中心點
            if (myTargetStandOnTransform != null && updateMesh > 0 && isInitPrefabPos == false)
            {
                myTargetStandOnTransform.localPosition = StandOnTargetPosition();
                isInitPrefabPos = true;

            }
            // 初始化頂點和面陣列
            if (topFace == null)
            {
                InitializeVerticesAndFaces();
                Debug.Log("Initialize Vertices And Faces.");
                Initialize = true;
            }
            //初次更新mesh
            if (Initialize == true)
            {
                UpdateMesh();
                Initialize = false;
                updateMesh++;
                Debug.Log("Initialize and Update Mesh.");
            }
            //任何點有變動時更新mesh的頂點
            if (!AreVerticesEqual(this.vertices, privousVertices) && updateMesh > 0)
            {
                InitializeVerticesAndFaces();
                UpdateMesh();
                privousVertices = vertices;
                Debug.Log("Change verticles UpdateMesh!");
            }

            //角度改變時更新mesh的頂點
            if (previousAngle != angle && updateMesh > 0)
            {
                Vector3 scale = transform.localScale;
                // 計算最大允許角度以防止幾何體翻轉
                // 當本地X位移量為1時（從-0.5到0.5），達到極限
                // displacementX = (scale.y / scale.x) * tan(angle) <= 1  =>  tan(angle) <= scale.x / scale.y
                float maxAngle = 40f; // 預設為屬性範圍最大值
                if (scale.y > 0 && !Mathf.Approximately(scale.x, 0))
                {
                    maxAngle = Mathf.Atan(scale.x / scale.y) * Mathf.Rad2Deg;
                }

                // 如果新角度超過最大值，則鎖定回上一個有效角度
                if (angle > maxAngle)
                {
                    // 使用延遲呼叫來重設值，避免在Inspector更新迴圈中直接修改而產生問題
                    EditorApplication.delayCall += () => 
                    {
                        if (this != null) angle = previousAngle;
                    };
                }
                else // 角度有效，正常更新
                {
                    AngleChanged = true;
                    InitializeVerticesAndFaces();
                    UpdateMesh();
                    PrefabWhenAngle();
                    offsetOnPlaneY = GetVerticalDistanceToLine();
                    previousAngle = angle;
                }
            }
            //縮放時更新worldVertices
            if (previousScales != transform.localScale)
            {
                isScale = true;
                InitializeVerticesAndFaces();
                UpdateMesh();

                //子物件Scale不受父物件影響
                PrefabWhenScale("TargetStandOn", targetStandOnScale);
                PrefabWhenScale("Back", myBackPrefabScale);
                offsetOnPlaneY = GetVerticalDistanceToLine();
                previousScales = transform.localScale;
            }
            //旋轉時更新worldVertices
            if (previousRotation != transform.rotation)
            {
                isRotate = true;
                InitializeVerticesAndFaces();
                UpdateMesh();
                PrefabWhenRotate();
                previousRotation = transform.rotation;
            }
            //位移時更新worldVertices
            if (previousPosition != transform.position)
            {
                isMove = true;
                InitializeVerticesAndFaces();
                UpdateMesh();
                previousPosition = transform.position;
            }
            //如果plane在已經掛載的狀況下移動，更新y值
            if (plane != null)
            {
                float currentY = plane.transform.position.y;
                if (!Mathf.Approximately(currentY, lastPlaneY))
                {
                    standOnPlaneY = currentY;
                    lastPlaneY = currentY;
                }
            }          
        }
        /// <summary>
        /// 初始化或更新頂點和面陣列
        /// </summary>
        void InitializeVerticesAndFaces()
        {
            //第一次初始化頂點
            if (updateMesh == 0)
            {
                // 初始化頂點
                vertices = new Vector3[]
                {
                    new Vector3(-0.5f, -0.5f, 0.5f),   // 下方(左上)
                    new Vector3(-0.5f, -0.5f, -0.5f),  // 下方(左下)
                    new Vector3(0.5f, -0.5f, -0.5f),   // 下方(右下)
                    new Vector3(0.5f, -0.5f, 0.5f),    // 下方(右上)
                    new Vector3(-0.5f, 0.5f, 0.5f), // 上方(左上)
                    new Vector3(-0.5f, 0.5f, -0.5f),// 上方(左下)
                    new Vector3(0.5f, 0.5f, -0.5f), // 上方(右下)
                    new Vector3(0.5f, 0.5f, 0.5f)   // 上方(右上)
                };
                privousVertices = (Vector3[])vertices.Clone();
                UpdateWorldVertices();
            }

            //調整改變角度後投影到的新頂點
            if (AngleChanged == true)
            {
                Vector3[] newVertices = AngleToPoint(angle);
                vertices[4] = newVertices[0];
                vertices[5] = newVertices[1];
                UpdateWorldVertices();
                privousVertices = (Vector3[])vertices.Clone();
                AngleChanged = false;
            }
            //縮放或旋轉時更新頂點
            if (isScale || isRotate || isMove)
            {
                UpdateWorldVertices();
                privousVertices = (Vector3[])vertices.Clone();
                isScale = false;
                isRotate = false;
                isMove = false;
            }
            else
            {
                //Debug.Log("No vertices data!");
            }

            // 面陣列定義
            bottomFace = new Vector3[4] { vertices[0], vertices[1], vertices[2], vertices[3] };
            leftFace = new Vector3[4] { vertices[4], vertices[5], vertices[1], vertices[0] };
            rightFace = new Vector3[4] { vertices[6], vertices[7], vertices[3], vertices[2] };
            frontFace = new Vector3[4] { vertices[5], vertices[6], vertices[2], vertices[1] };
            backFace = new Vector3[4] { vertices[7], vertices[4], vertices[0], vertices[3] };
            topFace = new Vector3[4] { vertices[6], vertices[5], vertices[4], vertices[7] };

            faces = new Vector3[][]
            {
                bottomFace, topFace, leftFace, rightFace, frontFace, backFace
            };
        }
        /// <summary>
        /// 更新Mesh
        /// </summary>
        void UpdateMesh()
        {
            if (faces != null)
            {
                if (myMeshFilter == null)
                {
                    Debug.Log("MeshFilter not found.");
                    return;
                }

                if (myMeshFilter.sharedMesh == null)
                {
                    myMeshFilter.sharedMesh = new Mesh
                    {
                        name = "InstanceMesh"
                    };
                }
                else
                {
                    myMeshFilter.sharedMesh = Instantiate(myMeshFilter.sharedMesh);
                }

                Mesh mesh = myMeshFilter.sharedMesh;
                mesh.Clear(); // 清除之前的内容

                List<Vector3> combinedVertices = new List<Vector3>();
                List<int> combinedTriangles = new List<int>();

                for (int i = 0; i < faces.Length; i++)
                {
                    Vector3[] faceVertices = faces[i];
                    int offset = combinedVertices.Count;
                    combinedVertices.AddRange(faceVertices);

                    // 根據面生成三角形
                    combinedTriangles.AddRange(new int[]
                    {
                        offset, offset + 1, offset + 2, // 第一個三角形
                        offset + 2, offset + 3, offset  // 第二個三角形
                    });
                }

                mesh.vertices = combinedVertices.ToArray();
                mesh.triangles = combinedTriangles.ToArray();
                mesh.RecalculateNormals();
            }
        }
        /// <summary>
        /// 檢查頂點是否更動
        /// </summary>
        /// <param name="current"></param>
        /// <param name="previous"></param>
        /// <returns></returns>
        private bool AreVerticesEqual(Vector3[] current, Vector3[] previous)
        {
            if (current == null || previous == null || current.Length != previous.Length)
                return false;

            for (int i = 0; i < current.Length; i++)
            {
                if (current[i] != previous[i])
                    return false;
            }
            return true;
        }
        /// <summary>
        /// 計算調整角度後，下方點投影到上方面的點位置
        /// </summary>
        /// <param name="angle"></param>
        /// <returns></returns>
        private Vector3[] AngleToPoint(float angle)
        {
            Vector3 scale = transform.localScale;

            // 防止X軸縮放為零
            if (Mathf.Approximately(scale.x, 0))
            {
                // 返回原始頂點位置以避免錯誤
                return new Vector3[] {
                    new Vector3(-0.5f, 0.5f, 0.5f),
                    new Vector3(-0.5f, 0.5f, -0.5f)
                };
            }

            // (scale.y / scale.x) * tan(angle)
            float displacementX = (scale.y / scale.x) * Mathf.Tan(angle * Mathf.Deg2Rad);

            // 原始頂點的初始X座標是 -0.5
            float newX = -0.5f + displacementX;

            // 限制X座標，防止幾何體翻轉或與右側交叉 (右側X座標為 0.5)
            newX = Mathf.Clamp(newX, -0.5f, 0.5f);

            // 創建新的頂點位置
            Vector3 newPoint4 = new Vector3(newX, 0.5f, 0.5f);
            Vector3 newPoint5 = new Vector3(newX, 0.5f, -0.5f);

            return new Vector3[] { newPoint4, newPoint5 };
        }
        /// <summary>
        /// 確定世界座標
        /// </summary>
        void UpdateWorldVertices()
        {
            for (int i = 0; i < vertices.Length; i++)
            {
                //局部空間頂點，先縮放，再旋轉，最後平移到世界座標
                Vector3 localVertex = vertices[i];
                Vector3 scaledVertex = Vector3.Scale(localVertex, transform.localScale); // 縮放
                Vector3 rotatedVertex = transform.rotation * scaledVertex; // 旋轉
                worldVertices[i] = rotatedVertex + transform.position; // 平移到世界座標
            }
        }
        /// <summary>
        /// 取德standOnTarget這個prefab在上面點位的中心位置
        /// </summary>
        /// <returns></returns>
        private Vector3 StandOnTargetPosition()
        {
            Vector3 center = (vertices[4] + vertices[5] + vertices[6] + vertices[7]) / 4;
            return center;
        }
        private void PrefabWhenScale(string PrefabName, Vector3 PrefabScale)
        {
            //保持prefab的localScale不被父物件影響
            Transform myTargetStandOnTransform = transform.Find(PrefabName);
            if (myTargetStandOnTransform != null && isInitPrefabPos)
            {
                //只保留小數點後2位
                myTargetStandOnTransform.localScale = new Vector3(
                    (float)(Math.Round((PrefabScale.x / transform.localScale.x) * 100f) / 100f),
                    (float)(Math.Round((PrefabScale.y / transform.localScale.y) * 100f) / 100f),
                    (float)(Math.Round((PrefabScale.z / transform.localScale.z) * 100f) / 100f)
                );
            }
        }
        private void PrefabWhenRotate()
        {
            //有改變位置或旋轉時，更新中心點位置
            Transform myTargetStandOnTransform = transform.Find("TargetStandOn");
            if (myTargetStandOnTransform != null)
            {
                myTargetStandOnTransform.localPosition = StandOnTargetPosition();
            }
        }
        private void PrefabWhenAngle()
        {
            //角度改變時，更新prefab的位置
            Transform myTargetStandOnTransform = transform.Find("TargetStandOn");
            if (myTargetStandOnTransform != null)
            {
                myTargetStandOnTransform.localPosition = StandOnTargetPosition();
            }
        }
        
        /// <summary>
        /// 取得垂直距離，算出offsetOnPlaneY的數值
        /// </summary>
        /// <returns></returns>
        private float GetVerticalDistanceToLine()
        {
            Vector3 lineDirection = vertices[5] - vertices[4];
            lineDirection.Normalize();
            
            var target = transform.Find("TargetStandOn");
            if (target == null)
            {
                Debug.LogWarning("TargetStandOn not found.");
                return 0f; 
            }
            Vector3 targetPosition = target.localPosition;
            Vector3 targetToLineStart = targetPosition - vertices[4];
            float distance = Vector3.Cross(lineDirection, targetToLineStart).magnitude;
            return distance;
        }

        private string myClimbIconPath = "Editor/Design/Icon/climbWall.png";
        void OnDrawGizmos()
        {
            //繪製點
            for (int i = 0; i < worldVertices.Length; i++)
            {
                Gizmos.color = green;
                Gizmos.DrawSphere(worldVertices[i], 0.03f);
            }
            //繪製角度面(綠)
            Vector3[] angleFace = new Vector3[]
            {
                worldVertices[0], worldVertices[1],worldVertices[5], worldVertices[4]
            };
            Gizmos.color = green;
            Gizmos.DrawLineStrip(angleFace, true);

            //繪製其他面(紫)
            Vector3[] faceFront = new Vector3[]
            {
                worldVertices[3], worldVertices[2],worldVertices[6], worldVertices[7]
            };
            Vector3[] faceLeft = new Vector3[]
            {
                worldVertices[5], worldVertices[6], worldVertices[2], worldVertices[1]
            };
            Vector3[] faceRight = new Vector3[]
            {
                worldVertices[0], worldVertices[3],worldVertices[7], worldVertices[4]
            };
            Gizmos.color = purple;
            Gizmos.DrawLineStrip(faceFront, true);
            Gizmos.DrawLineStrip(faceLeft, false);
            Gizmos.DrawLineStrip(faceRight, false);

            //在離視角一定距離後再顯示icon
            CameraToSceneView();
            if (toDistance >= closeDistance)
            {
                IconPathToPackage(myClimbIconPath, transform.position);
            }
        }

#endif
    }
}