using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using System.Linq;

#if UNITY_EDITOR
using UnityEditor;
#endif

namespace CHTLand.SDK.Module.Component{
    /// <summary>
    /// Custom NPC 元件
    /// </summary>
    [ExecuteInEditMode] 
    public class ChtDialogComponentWithCustomNPC : ChtBaseComponent
    { 
        [SerializeField] 
        private string dialogCustomComponentID;
        public string DialogCustomComponentID => dialogCustomComponentID;
       
       //Type of component
       [HideInInspector]
        ComponentType type = ComponentType.ChtDialogComponentWithCustomNPC;
        public GameObject myNpcModel;
        public string myNpcType = "Custom";
        public bool isWalking = false;
        public float WalkingSpeed = 1.0f;
        private RenderTexture myNpcRenderTexture;
        [SerializeField] public List<Vector3> pathPositions = new List<Vector3>(); 
        
        protected override void Awake()
        {
            base.Awake();
            //Debug.LogWarning($"Chest Component ID : {ObjectID}");

            //建立Render Texture給NpcCamera的Target Texture與 NpcCameraView/raw image Texture 
            myNpcRenderTexture = new RenderTexture(256, 256, 24)
            {
                antiAliasing = 1,
                useMipMap = false
            };
        }
        void Start()
        {
            if (Application.isPlaying && CHTLandProxy.componentManager != null)
            {
                CHTLandProxy.componentManager.InitDialogComponentWithCustomNPC(this); 
            }
            Debug.Log(type.ToString());
            Debug.Log("My NpcType: " + myNpcType);
        }
        
        #if UNITY_EDITOR

        
        private Transform myNpcCameraViewtransform;
        private Vector3 myNpcCameraViewPostion = new Vector3(-0.67f,0.715f,0.0f);
        private Quaternion myNpcCameraViewRotation = Quaternion.Euler(0,180,0); 
        private Vector3 myNpcColliderSize = new Vector3(1.0f,1.87f,1.0f);
        private Vector3 mynpcColliderCenter = new Vector3(0.0f,0.945f,0f);
        private Vector3 mySpiritOffset = new Vector3(-0.5f, 1.5f, 0);
        private GameObject priviousNpcModel;
        private bool isEnteringPlayMode = true;
        private bool allowAlert = true;
        private string NpcCameraViewPath = "Editor/Design/Prefab/NpcCameraView.prefab";
        private string NpcSpiritPath = "Editor/Design/Prefab/Spirit.prefab";
        private int previousChildCount = 0;
 
        //public bool BackToDefault;


        protected override void OnValidate()
        {
            base.OnValidate();
            if (!Application.isPlaying)
            {
                // 同步 base 的 objectID 給 seatComponentID
                dialogCustomComponentID = objectID;
                UnityEditor.EditorUtility.SetDirty(this); // 確保 Inspector 顯示更新
                InitCopyPrefab("Spirit", HideFlags.HideInHierarchy, "SpiritObject");
                InitCopyPrefab("NpcCameraView", HideFlags.HideInHierarchy); 
            }
        }

        void OnEnable()
        {
            EditorApplication.update += OnEditorUpdate;
            EditorApplication.playModeStateChanged += OnPlayModeStateChanged;
        }

        void OnDisable()
        {
            EditorApplication.update -= OnEditorUpdate;
            EditorApplication.playModeStateChanged -= OnPlayModeStateChanged;
        }

        void OnEditorUpdate()
        {
            // 如果正在Play或準備Play，就不要做任何事
            if (EditorApplication.isPlayingOrWillChangePlaymode)
            {
                return;
            }
            Camera cam = transform.GetComponentInChildren<Camera>();
            if (cam == null)
            {
                //創建NpcCamera後，設定NpcCamera上的RenderTexture
                InheritCreateCamera();
                InheritSetCameraTargetTexture(myNpcRenderTexture);
            }
            else
            {
                cam.hideFlags = HideFlags.NotEditable;
            }

            //確定有camera與myNpcModel,再新增NpcCamerView
            if (transform.GetComponentInChildren<Camera>() != null && myNpcModel != null && transform.Find("NpcCameraView") == null)
            {
                FindPrefab(NpcCameraViewPath);

                //設定NpcCameraView的位移
                myNpcCameraViewtransform = transform.Find("NpcCameraView");
                if (myNpcCameraViewtransform != null)
                {
                    RectTransform npcCameraViewRectTransform = myNpcCameraViewtransform.GetComponent<RectTransform>();
                    npcCameraViewRectTransform.localPosition = myNpcCameraViewPostion;
                    npcCameraViewRectTransform.localRotation = Quaternion.Euler(0.0f, 180.0f, 0.0f);
                    myNpcCameraViewtransform.rotation = myNpcCameraViewRotation;
                    InheritLockAllComponents(myNpcCameraViewtransform.gameObject);
                    Debug.Log("Find NpcCameraView");
                }
                else
                {
                    Debug.Log("Not Find NpcCameraView");
                }

                MySetRawImageTexture(myNpcRenderTexture);
            }

            //把myNpcModel設為子物件
            if (myNpcModel != null)
            {
                myNpcModel.transform.parent = this.transform;
                myNpcModel.transform.localPosition = Vector3.zero;

                //把myNpcModel所有子物件的Layer設為NPC
                InheritSetLayerRecursively(myNpcModel, LayerMask.NameToLayer("NPC"));

                //設定Npc Collider的大小
                BoxCollider npcCollider = this.GetComponent<BoxCollider>();
                if (npcCollider == null)
                {
                    CreateCollider();
                    npcCollider = this.GetComponent<BoxCollider>();
                    if (npcCollider != null)
                    {
                        npcCollider.size = myNpcColliderSize;
                        npcCollider.center = mynpcColliderCenter;
                    }
                }
                
                // 創建spirit
                if (transform.Find("SpiritObject") == null)
                {
                    // 創建一個空物件作為Spirit的父物件
                    GameObject spiritParentObj = new GameObject();
                    spiritParentObj.transform.parent = transform;
                    spiritParentObj.transform.localPosition = mySpiritOffset;
                    spiritParentObj.name = "SpiritObject";

                    FindPrefab(NpcSpiritPath);
                    var spiritPrefab = transform.Find("Spirit");
                    if (spiritPrefab != null)
                    {
                        spiritPrefab.transform.parent = spiritParentObj.transform;
                        spiritPrefab.transform.localPosition = Vector3.zero;
                        spiritPrefab.hideFlags = HideFlags.HideInHierarchy;
                    }
                }  
            }

            //確切在editor mode時候才ShowAlert
            if (isEnteringPlayMode == false && allowAlert == true)
            {
                if (myNpcModel != null && myNpcModel != priviousNpcModel)
                {
                    MyShowAlert();
                    priviousNpcModel = myNpcModel;
                }
                else
                {
                    //Debug.Log("NO NPC Model!");
                }
            }
            if (isWalking == true)
            {
                Transform myTargetStandOnTransform = transform.Find("Path");
                if (myTargetStandOnTransform == null)
                {
                    GameObject targetParent = new GameObject();
                    targetParent.transform.parent = transform;
                    targetParent.name = "Path";

                    //default path child 3 localPosition
                    var defaultPosition = new Vector3[]{
                        Vector3.zero,
                        new Vector3(2, 0, 0),
                        new Vector3(2, 0, -2)
                    };

                    //create default 3 path
                    for (int i = 0; i < 3; i++)
                    {
                        FindPrefab(TargetStandOn);
                        myTargetStandOnTransform = transform.Find("TargetStandOn");
                        myTargetStandOnTransform.transform.parent = targetParent.transform;
                        myTargetStandOnTransform.localPosition = defaultPosition[i];
                    }
                }
                else
                {
                    Transform path = transform.Find("Path");
                    if (previousChildCount != path.childCount)
                    {
                        RenameChildObjects();
                        previousChildCount = path.childCount;
                    }
                    pathPositions = GetPathPositions();
                    //Debug.Log(string.Join(",", pathPositions.Select(p => p.ToString())));
                }
            }
            else
            {
                //isWalking back to false, init pathPositions
                pathPositions = GetPathPositions();
                var paths = transform.Find("Path");
                if (paths != null)
                {
                    DestroyImmediate(paths.gameObject);
                }
            }
        

        }
        private void OnPlayModeStateChanged(PlayModeStateChange state)
        {
            if (state == PlayModeStateChange.ExitingEditMode)
            {
                isEnteringPlayMode = true;
                allowAlert = false;
            }
            else if (state == PlayModeStateChange.EnteredEditMode)
            {
                isEnteringPlayMode = true;
                allowAlert = false;
                
            }
            else if (state == PlayModeStateChange.EnteredPlayMode)
            {
                isEnteringPlayMode = true;
                allowAlert = false;
            }
            else if (state == PlayModeStateChange.ExitingPlayMode)
            {
                isEnteringPlayMode = true;
                allowAlert = false;
            }
            else //只有完全進入editor mode的時候才觸發alert
            {
                isEnteringPlayMode = false;
                EditorApplication.delayCall += () => allowAlert = true;
                
            }

        }
        private List<Vector3> GetPathPositions()
        {
            var pathes = transform.Find("Path");
            if (isWalking == true && pathes != null)
            {
                return pathes.Cast<Transform>().Select(p => p.position).ToList();
            }      
            return new List<Vector3>();
        }

        void OnDrawGizmos()
        {
            //Line Color
            Gizmos.color = purple;
            //Font Style
            GUIStyle style = new GUIStyle();
            style.normal.textColor = purple;
            style.fontStyle = FontStyle.Bold;
            style.fontSize = 20;
            style.contentOffset = new Vector2(15 , -30);      

            var path = transform.Find("Path");
            if (path != null && path.childCount > 0)
            {
                for (int i = 0; i < path.childCount; i++)
                {
                    var pointA = path.GetChild(i); 
                    Handles.Label(pointA.position, $"{i + 1}", style); 

                    if (i + 1 < path.childCount)
                    {
                        var pointB = path.GetChild(i + 1);
                        Gizmos.DrawLine(pointA.position, pointB.position);
                    }
                }
            }
        }
        
        [ContextMenu("Show Alert")]
        private void MyShowAlert()
        {
            // 顯示一個簡單的對話框
            EditorUtility.DisplayDialog("Alert", myNpcModel.name + " will become child of " + gameObject.name, "OK");
            Debug.Log("Alert!!!");
        }

        #endif
    }
}