using System.Collections;
using System.Collections.Generic;
using UnityEngine;

#if UNITY_EDITOR
using UnityEditor;
#endif

namespace CHTLand.SDK.Module.Component
{
    /// <summary>
    /// 圖片影片輪播元件
    /// </summary>
    [ExecuteInEditMode]
    public class ChtFrameComponent : ChtBaseComponent
    {
        [SerializeField] 
        private string frameComponentID;
        public string FrameComponentID => frameComponentID;

        //Type of component
        [HideInInspector]
        ComponentType type = ComponentType.ChtFrameComponent;
        protected override void Awake()
        {
            base.Awake();
            //Debug.LogWarning($"Chest Component ID : {ObjectID}");
        }
        void Start()
        {
            if (Application.isPlaying && CHTLandProxy.componentManager != null)
            {
                CHTLandProxy.componentManager.InitFrameComponent(this);
            }
            Debug.Log(type.ToString());
        }
        public enum FrameOption
        {
            Video,
            Image,
            //Carousel
        }
        public FrameOption selectedFrameOption;

#if UNITY_EDITOR
        private Vector3 myBackIconPosition = new Vector3(0.0f, 0.0f, 0.01f);
        private Vector3 myFrontIconPosition = new Vector3(0.0f, 0.0f, -0.01f);
        private Vector3 myIconScale = new Vector3(0.294f, 0.5f, 0.5f);
        private bool isDrawFarGizmos = false;
        private string myFrontIconPath = "Editor/Design/Prefab/Front.prefab";
        private string myBackIconPath = "Editor/Design/Prefab/Back.prefab";

        protected override void OnValidate()
        {
            base.OnValidate();
            if (!Application.isPlaying)
            {
                // 同步 base 的 objectID 給 seatComponentID
                frameComponentID = objectID;
                UnityEditor.EditorUtility.SetDirty(this); // 確保 Inspector 顯示更新
            }
        }

        void OnEnable()
        {
            EditorApplication.update += OnEditorUpdate;
            componentType = ComponentType.ChtFrameComponent;

        }

        void OnDisable()
        {
            EditorApplication.update -= OnEditorUpdate;
        }

        void OnEditorUpdate()
        {
            // 如果正在Play或準備Play，就不要做任何事
            if (EditorApplication.isPlayingOrWillChangePlaymode)
            {
                return;
            }

            MeshFilter meshFilter = gameObject.GetComponent<MeshFilter>();
            MeshRenderer meshRenderer = gameObject.GetComponent<MeshRenderer>();
            BoxCollider collider = gameObject.GetComponent<BoxCollider>();
            
            if (meshFilter == null)
            {
                CreateMeshFilter("Quad");
            }
            if (meshRenderer == null)
            {
                CreateRenderer();
            }

            if (meshFilter != null && meshRenderer != null)
            {
                meshFilter.hideFlags = HideFlags.NotEditable;
                meshRenderer.hideFlags = HideFlags.NotEditable;
            }
            
            if (collider == null)
            {
                CreateCollider();
            }

            if (transform.localScale != previousScale)
            {
                LockScale();
                previousScale = transform.localScale;

                //保持子物件icon的比例
                if (transform.Find("Front") != null && transform.Find("Back") != null)
                {
                    var transformFrontIcon = transform.Find("Front");
                    var transformBacklIcon = transform.Find("Back");

                    var x = transform.position.x;

                    transformFrontIcon.localScale = myIconScale;
                    transformBacklIcon.localScale = myIconScale;
                }
            }

            if (meshFilter != null && meshRenderer != null)
            {
                Vector3 detectCollider = new Vector3(transform.lossyScale.x, transform.lossyScale.y, 0.01f);
                DetectCollisions(default,default,detectCollider);
            }

            Transform iconFront = transform.Find("Front");
            Transform iconBack = transform.Find("Back");
            if (iconFront == null)
            {
                FindPrefab(myFrontIconPath);
                iconFront = transform.Find("Front");
                iconFront.parent = this.transform;
                iconFront.localPosition = myFrontIconPosition;
                iconFront.localScale = myIconScale;
                InheritLockAllComponents(iconFront.gameObject);
            }

            if (iconBack == null)
            {
                FindPrefab(myBackIconPath);
                iconBack = transform.Find("Back");
                iconBack.parent = this.transform;
                iconBack.localPosition = myBackIconPosition;
                iconBack.localScale = myIconScale;
                InheritLockAllComponents(iconBack.gameObject);
            }

            //計算物件與視角Camera的距離
            CameraToSceneView();
            Transform front = transform.Find("Front");
            Transform back = transform.Find("Back");
            //Debug.Log(toDistance.ToString());

            if (toDistance >= closeDistance)
            {
                front.gameObject.SetActive(false);
                back.gameObject.SetActive(false);
                isDrawFarGizmos = true;
            }
            else if (toDistance <= closeDistance)
            {
                front.gameObject.SetActive(true);
                back.gameObject.SetActive(true);
                isDrawFarGizmos = false;
            }
        }

        private string imageIconPath = "Editor/Design/Icon/image.png";
        private string videoIconPath = "Editor/Design/Icon/video.png";
        private string carouselIconPath = "Editor/Design/Icon/carousel.png";

        void OnDrawGizmos()
        {
            //Purple
            Gizmos.color = new UnityEngine.Color(112.0f / 255.0f, 100.0f / 255f, 154.0f / 255.0f, 210.0f / 255.0f);

            //使用cube的大小設定gizmos的矩陣
            Gizmos.matrix = Matrix4x4.TRS(transform.position, transform.rotation, new Vector3(transform.localScale.x, transform.localScale.y, 0.001f));
            Gizmos.DrawWireCube(Vector3.zero, Vector3.one);

            if (isDrawFarGizmos == true)
            {
                MyDrawGizmos();
            }
        }
        private void MyDrawGizmos()
        {
            //切換icon
            if (selectedFrameOption == FrameOption.Image)
            {
                IconPathToPackage(imageIconPath, transform.position);
            }
            else if (selectedFrameOption == FrameOption.Video)
            {
                IconPathToPackage(videoIconPath, transform.position);
            }
            /*
            else if (selectedFrameOption == FrameOption.Carousel)
            {
                IconPathToPackage(carouselIconPath, transform.position);
            }
            */
        }

#endif
    }
}
