

using System.Collections.Generic;
using UnityEngine;

/*Note:
1.多人連線刷新率為:48 tick rate
於任何平台與裝置皆為一致，意味著運行及時且激烈操作，需評估影響
2.於同一個Room Name下的使用者，才會看到對方且同步所有場景狀態
3.於同一個Room Name下的使用者，且同個頻道下，才會聽到對方語音
4.第一個進入房間或創建者，為Master腳色，有權限可以對同步參數(參考IMultiActManager)修改
5.執行搜尋房間後，才能取得房間列表
*/

namespace CHTLand.SDK.Module.Multiple
{
    /// <summary>
    /// 進入房間狀態定義
    /// RndomJoin : 隨機進入一個房間
    /// Create : 創建一個房間
    /// Specify : 指定一個已知房間
    /// </summary>
    public enum ConnectType
    {
        RandomJoin,
        Create,
        Specify,
    }

    /// <summary>
    /// 連線狀態定義
    /// </summary>
    public enum ConnectState
    {
        Connecting,
        Connected,
        Disconnected
    }

    /// <summary>
    /// 語音發送狀態定義
    /// TotalArea : 全區域都可以聽到說話語音
    /// PartArea_5M : 自身範圍5公尺內才可以聽到聲音
    /// </summary>
    public enum VoiceType
    {
        TotalArea = 2,
        PartArea_5M = 3,
        Release = -1
    }

    /// <summary>
    /// 房間資訊
    /// Name:房間名稱，用於指定加入或顯示
    /// PlayerCount : 當前人數
    /// MaxPlayers : 上限人數
    /// IsVisible : 是否可以進入
    /// IsOpen : 是否開放尋找
    /// </summary>    
    public struct CHTSessionInfo
    {
        public string Name;
        public bool IsVisible;
        public bool IsOpen;
        public int PlayerCount;
        public int MaxPlayers;

        public CHTSessionInfo(string name = null, bool isVisible = false, bool isOpen = false, int playerCount = -1, int maxPlayers = -1)
        {
            Name = name;
            IsVisible = isVisible;
            IsOpen = isOpen;
            PlayerCount = playerCount;
            MaxPlayers = maxPlayers;
        }
    }


    /// <summary>
    /// 用於管理多人連線的腳本
    /// </summary>
    public interface IMultiConnectManager
    {
        /// <summary>
        /// 重新初始化，清除所有Callback與搜尋列表行為
        /// </summary>
        public void Initial();

        /// <summary>
        /// 指定房間進入，會被配對至相同房間
        /// connectType 若選 random，則不須指定房間名稱
        /// </summary>
        /// <param name="connectType">指定連接房間的型態，</param>
        /// <param name="name">指定房間名稱</param>
        void Connect(ConnectType connectType, string name = null);


        /// <summary>
        /// 離開房間回到大廳
        /// </summary>
        void DisConnect();

        /// <summary> 
        /// </summary>
        /// <param name="multiConnectCallback">用於收到連線事件的回傳</param>
        /// <param name="roomName">註明此連接房間的唯一識別名稱，他人僅透過此名稱才可進入</param>
        /// <param name="playerLimit">註明此房間的人數上限</param>
        //void Connect(IMultiConnectCallback multiConnectCallback, string roomName, int playerLimit);

        /// <summary>
        /// 開始指定搜尋房間列表
        /// </summary>
        void StartSearchRoom(MonoBehaviour mono);

        /// <summary>
        /// 房間列表資料
        /// </summary>
        /// <returns>若無則回傳null</returns>
        List<CHTSessionInfo> GetRoomList();


        /// <summary>
        /// 取得連線狀態資訊
        /// </summary>
        /// <returns></returns>
        ConnectState GetConnectState();

        /// <summary>
        /// 取得連線房間名稱
        /// </summary>
        /// <returns></returns>
        CHTSessionInfo GetConnectedRoom();

        /// <summary>
        /// 取得確認是否為房主(第一個進入)
        /// </summary>
        /// <returns></returns>
        bool CheckIsMaster();

        /// <summary>
        /// 切換語音放送與收聽頻道
        /// </summary>
        /// <param name="channel">切換到此頻道說話,並收聽此頻道說話</param>
        public void ChangeVoiceChannel(int channel);

        /// <summary>
        /// 修改語音放送方式
        /// </summary>
        /// <param name="voiceType">語音放送方式</param>
        public void ChangeVoiceType(VoiceType voiceType);

        /// <summary>
        /// 此房間將不再開放，隨機模式、指定模式皆無法進入
        /// 呼叫成功回傳true，失敗回傳false
        /// 僅Master呼叫有效，關閉當前房間
        /// </summary>
        public bool CloseCurrentRoom();

        /// <summary>
        /// 主動即時監看玩家資訊改變
        /// </summary>
        void RegisterCallback(IMultiConnectCallback callback);
    }

    /// <summary>
    /// 接收連線狀態改變的函式
    /// </summary>
    public interface IMultiConnectCallback
    {
        void OnConnectStateChanged(ConnectState connectState);
        //void OnGetRoomName(CHTSessionInfo chtSessionInfo);
        void OnPlayersChange(int count);
    }
}
