using System;
using UnityEngine;

namespace CHTLand.SDK.Module.Storage
{
    /// <summary>
    /// 錯誤類型列舉
    /// </summary>
    public enum SceneStorageError
    {
        NoMatchedKey,
        UnsupportedDataType,
        Network
    }

    /// <summary>
    /// 圖片類型列舉
    /// </summary>
    public enum SceneStorageImageType
    {
        JPG,
        PNG
    }

    /// <summary>
    /// SceneStorageManager介面
    /// </summary>
    public interface ISceneStorageManager
    {
        /// <summary>
        /// 取得完整Scene Data JSON字串，提供除錯使用。
        /// </summary>
        /// <param name="callback">實作OnLoadComplete(string result)取得JSON字串</param>
        void DumpAllData(ISceneStorageCallback callback);

        /// <summary>
        /// 清空全部Scene Data
        /// </summary>
        /// <param name="callback">實作OnLoadComplete()得到成功通知</param>
        void ClearAllData(ISceneStorageCallback callback);

        /// <summary>
        /// 儲存一筆 Scene Data
        /// </summary>
        /// <param name="key">資料鍵值</param>
        /// <param name="value">資料物件，支援string/int/float/string[]/int[]/float[]六種型態</param>
        /// <param name="callback">實作OnLoadComplete()得到成功通知</param>
        void SaveData(string key, object value, ISceneStorageCallback callback);

        /// <summary>
        /// 取得一筆 Scene Data
        /// </summary>
        /// <param name="key">資料鍵值</param>
        /// <param name="callback">實作OnLoadComplete(Object result)取得結果物件</param>
        void GetData(string key, ISceneStorageCallback callback);

        /// <summary>
        /// 刪除一筆 Scene Data
        /// </summary>
        /// <param name="key">資料鍵值</param>
        /// <param name="callback">實作OnLoadComplete()得到成功通知</param>
        void DeleteData(string key, ISceneStorageCallback callback);

        /// <summary>
        /// 上傳圖片
        /// </summary>
        /// <param name="bytes">圖片binary</param>
        /// <param name="type">圖片類型</param>
        /// <param name="callback">實作OnLoadComplete(string result)取得圖片檔名</param>
        void UploadImage(byte[] bytes, SceneStorageImageType type, ISceneStorageCallback callback);

        /// <summary>
        /// 下載圖片
        /// </summary>
        /// <param name="url">圖片url</param>
        /// <param name="callback">實作OnLoadComplete(Texture2D texture)取得圖片texture</param>
        void DownloadImage(string url, ISceneStorageCallback callback);
    }

    /// <summary>
    /// 實作Callback取得ISceneStorageManager方法執行結果
    /// </summary>
    public interface ISceneStorageCallback
    {
        /// <summary>
        /// 資料操作成功
        /// </summary>
        void OnLoadComplete() => Console.WriteLine("Default implementation.");

        /// <summary>
        /// 取得資料成功
        /// </summary>
        /// <param name="result">結果字串</param>
        void OnLoadComplete(string result) => Console.WriteLine("Default implementation.");

        /// <summary>
        /// 取得資料成功
        /// </summary>
        /// <param name="result">結果物件</param>
        void OnLoadComplete(object result) => Console.WriteLine("Default implementation.");

        /// <summary>
        /// 圖片載入成功
        /// </summary>
        /// <param name="texture">圖片紋理貼圖</param>
        void OnLoadComplete(Texture2D texture) => Console.WriteLine("Default implementation.");

        /// <summary>
        /// 資料操作失敗
        /// </summary>
        /// <param name="errorType">錯誤類型</param>
        /// <param name="networkErrorDetails">詳細資訊</param>
        void OnLoadFailed(SceneStorageError errorType, string networkErrorDetails);
    }
}