using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using CHTLand.SDK.Module;
using CHTLand.SDK.Module.Network;
using UnityEngine.Networking;
using System.Threading.Tasks;
using System;

public class TestNetworkManager : MonoBehaviour
{
    // Start is called before the first frame update
    async void Start()
    {
        IValidatedWebRequest request = CHTLandProxy.networkRequestManager.GetValidatedWebRequest();

        // 開始HTTP Get請求 (使用async)
        await GetHttpBinDataAsync(request);

        // 開始HTTP Post請求 (使用async)
        await PostHttpBinDataAsync(request);
        
        // 開始HTTP Post Form請求 (使用async)
        await PostFormDataAsync(request);
    }

    // 使用UnityWebRequest從httpbin.org獲取數據 (async Task版本)
    private async Task GetHttpBinDataAsync(IValidatedWebRequest validatedRequest)
    {
        string url = "https://httpbin.org/get";
        Debug.Log("開始GET請求: " + url);

        // 使用ValidatedWebRequest獲取UnityWebRequest物件
        UnityWebRequest webRequest = validatedRequest.Get(url);

        if (webRequest == null)
        {
            Debug.LogError("無法創建GET請求，URL可能不在允許的域名清單中或使用了非HTTPS協議");
            return;
        }

        // 發送請求 (使用async/await替代yield return)
        var operation = webRequest.SendWebRequest();

        // 等待請求完成
        while (!operation.isDone)
        {
            await Task.Yield();
        }

        // 檢查是否有錯誤
        if (webRequest.result != UnityWebRequest.Result.Success)
        {
            Debug.LogError("網絡請求錯誤: " + webRequest.error);
        }
        else
        {
            // 獲取並打印JSON響應
            string jsonResult = webRequest.downloadHandler.text;
            Debug.Log("GET請求成功! 收到的JSON響應 (async方式):");
            Debug.Log(jsonResult);

            // 可以進一步解析JSON數據
            // var resultData = JsonUtility.FromJson<YourDataType>(jsonResult);
        }

        // 釋放資源
        webRequest.Dispose();
    }

    // 使用UnityWebRequest向httpbin.org發送Post數據 (async Task版本)
    private async Task PostHttpBinDataAsync(IValidatedWebRequest validatedRequest)
    {
        string url = "https://httpbin.org/post";
        Debug.Log("開始POST請求: " + url);

        // 創建POST請求的數據
        var postData = new Dictionary<string, string>
        {
            { "name", "TestUser" },
            { "message", "Hello from CHTLand App" }
        };
        string jsonData = JsonUtility.ToJson(new SerializableDict(postData));

        // 使用ValidatedWebRequest獲取UnityWebRequest物件
        UnityWebRequest webRequest = validatedRequest.Post(url, jsonData, "application/json");

        if (webRequest == null)
        {
            Debug.LogError("無法創建Post請求，URL可能不在允許的域名清單中或使用了非HTTPS協議");
            return;
        }

        // 發送請求
        var operation = webRequest.SendWebRequest();

        // 等待請求完成
        while (!operation.isDone)
        {
            await Task.Yield();
        }

        // 檢查是否有錯誤
        if (webRequest.result != UnityWebRequest.Result.Success)
        {
            Debug.LogError("POST請求錯誤: " + webRequest.error);
        }
        else
        {
            // 獲取並打印JSON響應
            string jsonResult = webRequest.downloadHandler.text;
            Debug.Log("POST請求成功! 收到的JSON響應:");
            Debug.Log(jsonResult);
        }

        // 釋放資源
        webRequest.Dispose();
    }

    // 使用UnityWebRequest向httpbin.org發送FormData (async Task版本)
    private async Task PostFormDataAsync(IValidatedWebRequest validatedRequest)
    {
        string url = "https://httpbin.org/post";
        Debug.Log("開始POST Form請求: " + url);

        // 創建表單資料
        WWWForm formData = new WWWForm();
        formData.AddField("username", "TestUser");
        formData.AddField("email", "test@example.com");
        formData.AddField("message", "Hello from CHTLand App using form data!");

        // 使用ValidatedWebRequest獲取UnityWebRequest物件
        UnityWebRequest webRequest = validatedRequest.Post(url, formData);

        if (webRequest == null)
        {
            Debug.LogError("無法創建Post Form請求，URL可能不在允許的域名清單中或使用了非HTTPS協議");
            return;
        }

        // 發送請求
        var operation = webRequest.SendWebRequest();

        // 等待請求完成
        while (!operation.isDone)
        {
            await Task.Yield();
        }

        // 檢查是否有錯誤
        if (webRequest.result != UnityWebRequest.Result.Success)
        {
            Debug.LogError("POST Form請求錯誤: " + webRequest.error);
        }
        else
        {
            // 獲取並打印JSON響應
            string jsonResult = webRequest.downloadHandler.text;
            Debug.Log("POST Form請求成功! 收到的JSON響應:");
            Debug.Log(jsonResult);
        }

        // 釋放資源
        webRequest.Dispose();
    }

    // 為了序列化Dictionary用的輔助類
    [Serializable]
    private class SerializableDict
    {
        [Serializable]
        public class KeyValuePair
        {
            public string Key;
            public string Value;
        }

        public List<KeyValuePair> Items = new List<KeyValuePair>();

        public SerializableDict(Dictionary<string, string> dict)
        {
            foreach (var kvp in dict)
            {
                Items.Add(new KeyValuePair { Key = kvp.Key, Value = kvp.Value });
            }
        }
    }

    // Update is called once per frame
    void Update()
    {

    }
}
